<?php
defined('ABSPATH') || exit;
//VERSION = '4.7.1';
if (dej_opt('dej_show_wishlist', 1)) {
class DejavoWL
{
    protected static $products = [];
    protected static $settings = [];
    protected static $instance = null;

    public static function instance()
    {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    function __construct()
    {
        self::$settings = (array)get_option('DejavoWL_settings', []);

        delete_option('DejavoWL_localization');

        // add query var
        add_filter('query_vars', [$this, 'query_vars'], 1);
        add_action('init', [$this, 'init']);

        // menu
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_menu', [$this, 'admin_menu']);

        // my account
        if (self::get_setting('page_myaccount', 'yes') === 'yes') {
            add_filter('woocommerce_account_menu_items', [$this, 'account_items'], 99);
            add_action('woocommerce_account_wishlist_endpoint', [$this, 'account_endpoint'], 99);
        }

        // scripts
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);

        // quickview
        add_action('wp_ajax_wishlist_quickview', [$this, 'wishlist_quickview']);

        // add
        add_action('wp_ajax_wishlist_add', [$this, 'wishlist_add']);
        add_action('wp_ajax_nopriv_wishlist_add', [$this, 'wishlist_add']);

        // add to wishlist
        add_action('template_redirect', [$this, 'wishlist_add_by_link']);

        // added to cart
        if (self::get_setting('auto_remove', 'no') === 'yes') {
            add_action('woocommerce_add_to_cart', [$this, 'add_to_cart'], 10, 2);
        }

        // remove
        add_action('wp_ajax_wishlist_remove', [$this, 'wishlist_remove']);
        add_action('wp_ajax_nopriv_wishlist_remove', [$this, 'wishlist_remove']);

        // empty
        add_action('wp_ajax_wishlist_empty', [$this, 'wishlist_empty']);
        add_action('wp_ajax_nopriv_wishlist_empty', [$this, 'wishlist_empty']);

        // load
        add_action('wp_ajax_wishlist_load', [$this, 'wishlist_load']);
        add_action('wp_ajax_nopriv_wishlist_load', [$this, 'wishlist_load']);

        // load count
        add_action('wp_ajax_wishlist_load_count', [$this, 'wishlist_load_count']);
        add_action('wp_ajax_nopriv_wishlist_load_count', [$this, 'wishlist_load_count']);

        // fragments
        add_action('wp_ajax_DejavoWL_get_data', [$this, 'get_data']);
        add_action('wp_ajax_nopriv_DejavoWL_get_data', [$this, 'get_data']);

        // footer
        add_action('wp_footer', [$this, 'wp_footer']);


        // user login & logout
        add_action('wp_login', [$this, 'wp_login'], 10, 2);
        add_action('wp_logout', [$this, 'wp_logout']);


    }

    function query_vars($vars)
    {
        $vars[] = 'DejavoWL_id';

        return $vars;
    }

    function init()
    {
        // get key
        $key = isset($_COOKIE['DejavoWL_key']) ? sanitize_text_field($_COOKIE['DejavoWL_key']) : '#';

        // get products
        self::$products = self::get_ids($key);

        // rewrite
        if ($page_id = self::get_page_id()) {
            $page_slug = get_post_field('post_name', $page_id);

            if ($page_slug !== '') {
                add_rewrite_rule('^' . $page_slug . '/([\w]+)/?', 'index.php?page_id=' . $page_id . '&DejavoWL_id=$matches[1]', 'top');
                add_rewrite_rule('(.*?)/' . $page_slug . '/([\w]+)/?', 'index.php?page_id=' . $page_id . '&DejavoWL_id=$matches[2]', 'top');
            }
        }

        // my account page
        if (self::get_setting('page_myaccount', 'yes') === 'yes') {
            add_rewrite_endpoint('wishlist', EP_PAGES);
        }

        // shortcode
        add_shortcode('DejavoWL_list', [$this, 'shortcode_list']);


    }

    public static function get_settings()
    {
        return self::$settings;
    }

    public static function get_setting($name, $default = false)
    {
        if (!empty(self::$settings) && isset(self::$settings[$name])) {
            $setting = self::$settings[$name];
        } else {
            $setting = get_option('DejavoWL_' . $name, $default);
        }
        return $setting;
    }


    function add_to_cart($cart_item_key, $product_id)
    {
        $key = self::get_key();

        if ($key !== '#') {
            $products = self::get_ids($key);

            if (array_key_exists($product_id, $products)) {
                unset($products[$product_id]);
                update_option('DejavoWL_list_' . $key, $products);
                self::update_product_count($product_id, 'remove');
            }
        }
    }

    function wishlist_add_by_link()
    {
        if (!isset($_REQUEST['add-to-wishlist']) && !isset($_REQUEST['add_to_wishlist'])) {
            return false;
        }

        $key = self::get_key();
        $product_id = absint(isset($_REQUEST['add_to_wishlist']) ? (int)sanitize_text_field($_REQUEST['add_to_wishlist']) : 0);
        $product_id = absint(isset($_REQUEST['add-to-wishlist']) ? (int)sanitize_text_field($_REQUEST['add-to-wishlist']) : $product_id);

        if ($product_id) {
            if ($key !== '#' && $key !== 'DejavoWL') {
                $product = wc_get_product($product_id);
                $products = self::get_ids($key);

                if (!array_key_exists($product_id, $products)) {
                    // insert if not exists
                    $products = [
                            $product_id => [
                                'time' => time(),
                                'price' => is_a($product, 'WC_Product') ? $product->get_price() : 0,
                                'parent' => wp_get_post_parent_id($product_id) ?: 0,
                            ]
                        ] + $products;
                    update_option('DejavoWL_list_' . $key, $products);
                }
            }
        }

        // redirect to wishlist page
        wp_safe_redirect(self::get_url($key, true));
    }

    function wishlist_add()
    {
        $return = ['status' => 0];
        $key = self::get_key();

        if (($product_id = (int)sanitize_text_field($_POST['product_id'])) > 0) {
            if ($key === '#') {
                $return['status'] = 0;
                $return['notice'] = __('Please log in to use the Wishlist!', 'dejavo');
                $return['content'] = self::wishlist_content($key, esc_html__('There are no products on the Wishlist!', 'dejavo'));
            } else {
                $products = self::get_ids($key);

                if (!array_key_exists($product_id, $products)) {
                    // insert if not exists
                    $product = wc_get_product($product_id);
                    $products = [
                            $product_id => [
                                'time' => time(),
                                'price' => is_a($product, 'WC_Product') ? $product->get_price() : 0,
                                'parent' => wp_get_post_parent_id($product_id) ?: 0,
                            ]
                        ] + $products;
                    update_option('DejavoWL_list_' . $key, $products);
                    self::update_product_count($product_id, 'add');
                    $return['notice'] = esc_html__('{name} has been added to Wishlist.', 'dejavo');
                } else {
                    $return['notice'] = esc_html__('{name} is already in the Wishlist.', 'dejavo');
                }

                $return['status'] = 1;
                $return['count'] = count($products);
                $return['data'] = [
                    'key' => self::get_key(),
                    'ids' => self::get_ids(),
                ];

                if (self::get_setting('button_action', 'list') === 'list') {
                    $return['content'] = self::wishlist_content($key);
                }
            }
        } else {
            $product_id = 0;
            $return['status'] = 0;
            $return['notice'] = __('Have an error, please try again!', 'dejavo');
        }

        wp_send_json($return);
    }

    function wishlist_remove()
    {
        $return = ['status' => 0];
        $key = sanitize_text_field($_POST['key']);

        if (empty($key)) {
            $key = self::get_key();
        }

        if (($product_id = (int)sanitize_text_field($_POST['product_id'])) > 0) {
            if ($key === '#') {
                $return['notice'] = __('Please log in to use the Wishlist!', 'dejavo');
            } else {
                $products = self::get_ids($key);

                if (array_key_exists($product_id, $products)) {
                    unset($products[$product_id]);
                    update_option('DejavoWL_list_' . $key, $products);
                    self::update_product_count($product_id, 'remove');
                    $return['count'] = count($products);
                    $return['status'] = 1;
                    $return['notice'] = __('Product has been removed from the Wishlist.', 'dejavo');
                    $return['data'] = [
                        'key' => self::get_key(),
                        'ids' => self::get_ids(),
                    ];

                    if (empty($products)) {
                        $return['content'] = self::wishlist_content($key, __('There are no products on the Wishlist!', 'dejavo')) . '</div>';
                    }
                } else {
                    $return['notice'] = __('The product does not exist on the Wishlist!', 'dejavo');
                }
            }
        } else {
            $product_id = 0;
            $return['notice'] = __('Have an error, please try again!', 'dejavo');
        }

        wp_send_json($return);
    }

    function wishlist_empty()
    {
        $return = ['status' => 0];
        $key = sanitize_text_field($_POST['key']);

        if (empty($key)) {
            $key = self::get_key();
        }

        if ($key === '#') {
            $return['notice'] = __('Please log in to use the Wishlist!', 'dejavo');
        } else {
            if (($products = self::get_ids($key)) && !empty($products)) {
                foreach (array_keys($products) as $product_id) {
                    // update count
                    self::update_product_count($product_id, 'remove');
                }
            }

            // remove option
            update_option('DejavoWL_list_' . $key, []);
            $return['status'] = 1;
            $return['count'] = 0;
            $return['notice'] = __('All products have been removed from the Wishlist!', 'dejavo');
            $return['content'] = self::wishlist_content($key, __('There are no products on the Wishlist!', 'dejavo'));
            $return['data'] = [
                'key' => self::get_key(),
                'ids' => self::get_ids(),
            ];
        }


        wp_send_json($return);
    }

    function wishlist_load()
    {
        $return = ['status' => 0];
        $key = self::get_key();

        if ($key === '#') {
            $return['notice'] = __('Please log in to use Wishlist!', 'dejavo');
            $return['content'] = self::wishlist_content($key, __('There are no products on the Wishlist!', 'dejavo'));
        } else {
            $products = self::get_ids($key);
            $return['status'] = 1;
            $return['count'] = count($products);
            $return['content'] = self::wishlist_content($key);
            $return['data'] = [
                'key' => self::get_key(),
                'ids' => self::get_ids(),
            ];
        }


        wp_send_json($return);
    }

    function wishlist_load_count()
    {
        $return = ['status' => 0, 'count' => 0];
        $key = self::get_key();

        if ($key === '#') {
            $return['notice'] = __('Please log in to use Wishlist!', 'dejavo');
        } else {
            $products = self::get_ids($key);
            $return['status'] = 1;
            $return['count'] = count($products);
        }

        wp_send_json($return);
    }


    public static function wl_btn($id): string
    {
        $output = '';
        if ($id) {
            $class = 'DejavoWL-btn DejavoWL-btn-' . esc_attr($id);
            if (array_key_exists($id, self::$products) || in_array($id, array_column(self::$products, 'parent'))) {
                $class .= ' DejavoWL-added';
            }
            $output = '<a rel="nofollow" href="' . esc_url('?add-to-wishlist=' . $id) . '" class="' . esc_attr($class) . '" data-id="' . esc_attr($id) . '"><i class="dej-heart"></i></a>';
        }
        return $output;
    }

    function shortcode_list()
    {
        if (get_query_var('DejavoWL_id')) {
            $key = get_query_var('DejavoWL_id');
        } elseif (isset($_REQUEST['wid']) && !empty($_REQUEST['wid'])) {
            $key = sanitize_text_field($_REQUEST['wid']);
        } else {
            $key = self::get_key();
        }


        $share_url_raw = self::get_url($key, true);
        $share_url = urlencode($share_url_raw);
        $return_html = '<div class="DejavoWL-list">';
        $return_html .= self::get_items($key, 'table');
        $return_html .= '<div class="DejavoWL-actions">';


        $return_html .= '<div class="DejavoWL-share">';
        $return_html .= '<span class="DejavoWL-share-label">' . __('Share on:', 'dejavo') . '</span>';
        $return_html .= '<a class="DejavoWL-share-facebook" href="https://www.facebook.com/sharer.php?u=' . $share_url . '" target="_blank"><i class="dejbrand-facebook"></i></a>';
        $return_html .= '<a class="DejavoWL-share-twitter" href="https://twitter.com/share?url=' . $share_url . '" target="_blank"><i class="dejbrand-twitter"></i></a>';
        $return_html .= '<a class="DejavoWL-share-pinterest" href="https://pinterest.com/pin/create/button/?url=' . $share_url . '" target="_blank"><i class="dejbrand-pinterest"></i></a>';
        $return_html .= '<a class="DejavoWL-share-mail" href="mailto:?body=' . $share_url . '" target="_blank"><i class="dej-envelope"></i></a>';
        $return_html .= '</div><!-- /DejavoWL-share -->';


        $return_html .= '<div class="DejavoWL-copy">';
        $return_html .= '<span class="DejavoWL-copy-label">' . __('Wishlist link:', 'dejavo') . '</span>';
        $return_html .= '<span class="DejavoWL-copy-url"><input id="DejavoWL_copy_url" type="url" value="' . esc_attr($share_url_raw) . '" readonly/></span>';
        $return_html .= '<span class="DejavoWL-copy-btn"><input id="DejavoWL_copy_btn" type="button" value="' . __('Copy', 'dejavo') . '"/></span>';
        $return_html .= '</div><!-- /DejavoWL-copy -->';


        $return_html .= '</div><!-- /DejavoWL-actions -->';
        $return_html .= '</div><!-- /DejavoWL-list -->';

        return $return_html;
    }

    function register_settings()
    {
        register_setting('DejavoWL_settings', 'DejavoWL_settings');
    }

    function admin_menu()
    {
        add_submenu_page('future', __('Wishlist', 'dejavo'), __('Wishlist', 'dejavo'), 'manage_options', 'future-DejavoWL', array(
            $this,
            'admin_menu_content'
        ));
    }

    function admin_menu_content()
    {
        if (is_admin()) {
            ?>
            <div class="dejavo_settings_page wrap">
                <h1 class="dejavo_settings_page_title"><?php echo __('Wishlist', 'dejavo'); ?></h1>
                <?php if (isset($_GET['settings-updated']) && $_GET['settings-updated']) { ?>
                    <div class="notice notice-success is-dismissible">
                        <p><?php _e('Settings updated.', 'dejavo'); ?></p>
                    </div>
                <?php } ?>

                <div class="dejavo_settings_page_content">
                    <?php
                    if (isset($_REQUEST['settings-updated']) && (sanitize_text_field($_REQUEST['settings-updated']) === 'true')) {
                        flush_rewrite_rules();
                    }
                    $disable_unauthenticated = self::get_setting('disable_unauthenticated', 'no');
                    $auto_remove = self::get_setting('auto_remove', 'no');
                    $button_action = self::get_setting('button_action', 'list');
                    $message_position = self::get_setting('message_position', 'right-top');
                    $button_action_added = self::get_setting('button_action_added', 'popup');
                    $popup_position = self::get_setting('popup_position', 'center');
                    $link = self::get_setting('link', 'yes');
                    $show_price_change = self::get_setting('show_price_change', 'both');
                    $empty_button = self::get_setting('empty_button', 'yes');
                    $page_myaccount = self::get_setting('page_myaccount', 'yes');
                    $menu_action = self::get_setting('menu_action', 'open_popup');
                    ?>
                    <form method="post" action="options.php">
                        <table class="form-table">
                            <tr class="heading">
                                <th colspan="2">
                                    <?php _e('General', 'dejavo'); ?>
                                </th>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Disable the wishlist for unauthenticated users', 'dejavo'); ?></th>
                                <td>
                                    <select name="DejavoWL_settings[disable_unauthenticated]">
                                        <option value="yes" <?php selected($disable_unauthenticated, 'yes'); ?>><?php _e('Yes', 'dejavo'); ?></option>
                                        <option value="no" <?php selected($disable_unauthenticated, 'no'); ?>><?php _e('No', 'dejavo'); ?></option>
                                    </select>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Auto remove', 'dejavo'); ?></th>
                                <td>
                                    <select name="DejavoWL_settings[auto_remove]">
                                        <option value="yes" <?php selected($auto_remove, 'yes'); ?>><?php _e('Yes', 'dejavo'); ?></option>
                                        <option value="no" <?php selected($auto_remove, 'no'); ?>><?php _e('No', 'dejavo'); ?></option>
                                    </select>
                                    <span class="description"><?php _e('Auto remove product from the wishlist after adding to the cart.', 'dejavo'); ?></span>
                                </td>
                            </tr>


                            <tr>
                                <th scope="row"><?php _e('Action', 'dejavo'); ?></th>
                                <td>
                                    <select name="DejavoWL_settings[button_action]" class="DejavoWL_button_action">
                                        <option value="message" <?php selected($button_action, 'message'); ?>><?php _e('Show message', 'dejavo'); ?></option>
                                        <option value="list" <?php selected($button_action, 'list'); ?>><?php _e('Open wishlist popup', 'dejavo'); ?></option>
                                        <option value="no" <?php selected($button_action, 'no'); ?>><?php _e('Add to wishlist solely', 'dejavo'); ?></option>
                                    </select>
                                    <span class="description"><?php _e('Action triggered by clicking on the wishlist button.', 'dejavo'); ?></span>
                                </td>
                            </tr>
                            <tr class="DejavoWL_button_action_hide DejavoWL_button_action_message">
                                <th scope="row"><?php _e('Message position', 'dejavo'); ?></th>
                                <td>
                                    <select name="DejavoWL_settings[message_position]">
                                        <option value="right-top" <?php selected($message_position, 'right-top'); ?>><?php _e('right-top', 'dejavo'); ?></option>
                                        <option value="right-bottom" <?php selected($message_position, 'right-bottom'); ?>><?php _e('right-bottom', 'dejavo'); ?></option>
                                        <option value="fluid-top" <?php selected($message_position, 'fluid-top'); ?>><?php _e('center-top', 'dejavo'); ?></option>
                                        <option value="fluid-bottom" <?php selected($message_position, 'fluid-bottom'); ?>><?php _e('center-bottom', 'dejavo'); ?></option>
                                        <option value="left-top" <?php selected($message_position, 'left-top'); ?>><?php _e('left-top', 'dejavo'); ?></option>
                                        <option value="left-bottom" <?php selected($message_position, 'left-bottom'); ?>><?php _e('left-bottom', 'dejavo'); ?></option>
                                    </select>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Action (added)', 'dejavo'); ?></th>
                                <td>
                                    <select name="DejavoWL_settings[button_action_added]">
                                        <option value="popup" <?php selected($button_action_added, 'popup'); ?>><?php _e('Open wishlist popup', 'dejavo'); ?></option>
                                        <option value="page" <?php selected($button_action_added, 'page'); ?>><?php _e('Open wishlist page', 'dejavo'); ?></option>
                                    </select>
                                    <span class="description"><?php _e('Action triggered by clicking on the wishlist button after adding an item to the wishlist.', 'dejavo'); ?></span>
                                </td>
                            </tr>


                            <tr class="heading">
                                <th>
                                    <?php _e('Popup', 'dejavo'); ?>
                                </th>
                                <td>
                                    <?php _e('Settings for the wishlist popup.', 'dejavo'); ?>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Position', 'dejavo'); ?></th>
                                <td>
                                    <select name="DejavoWL_settings[popup_position]">
                                        <option value="center" <?php selected($popup_position, 'center'); ?>><?php _e('Center', 'dejavo'); ?></option>
                                        <option value="right" <?php selected($popup_position, 'right'); ?>><?php _e('Right', 'dejavo'); ?></option>
                                        <option value="left" <?php selected($popup_position, 'left'); ?>><?php _e('Left', 'dejavo'); ?></option>
                                    </select>
                                </td>
                            </tr>

                            <tr>
                                <th><?php _e('Link to individual product', 'dejavo'); ?></th>
                                <td>
                                    <select name="DejavoWL_settings[link]">
                                        <option value="yes" <?php selected($link, 'yes'); ?>><?php _e('Yes, open in the same tab', 'dejavo'); ?></option>
                                        <option value="yes_blank" <?php selected($link, 'yes_blank'); ?>><?php _e('Yes, open in the new tab', 'dejavo'); ?></option>
                                        <option value="yes_popup" <?php selected($link, 'yes_popup'); ?>><?php _e('Yes, open quick view popup', 'dejavo'); ?></option>
                                        <option value="no" <?php selected($link, 'no'); ?>><?php _e('No', 'dejavo'); ?></option>
                                    </select>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Show price change', 'dejavo'); ?></th>
                                <td>
                                    <select name="DejavoWL_settings[show_price_change]">
                                        <option value="no" <?php selected($show_price_change, 'no'); ?>><?php _e('No', 'dejavo'); ?></option>
                                        <option value="increase" <?php selected($show_price_change, 'increase'); ?>><?php _e('Increase only', 'dejavo'); ?></option>
                                        <option value="decrease" <?php selected($show_price_change, 'decrease'); ?>><?php _e('Decrease only', 'dejavo'); ?></option>
                                        <option value="both" <?php selected($show_price_change, 'both'); ?>><?php _e('Both increase and decrease', 'dejavo'); ?></option>
                                    </select>
                                    <span class="description"><?php _e('Show price change since a product was added.', 'dejavo'); ?></span>
                                </td>
                            </tr>


                            <tr>
                                <th scope="row"><?php _e('Empty wishlist button', 'dejavo'); ?></th>
                                <td>
                                    <select name="DejavoWL_settings[empty_button]">
                                        <option value="yes" <?php selected($empty_button, 'yes'); ?>><?php _e('Yes', 'dejavo'); ?></option>
                                        <option value="no" <?php selected($empty_button, 'no'); ?>><?php _e('No', 'dejavo'); ?></option>
                                    </select>
                                    <span class="description"><?php _e('Show empty wishlist button on the popup?', 'dejavo'); ?></span>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Continue shopping link', 'dejavo'); ?></th>
                                <td>
                                    <input type="url" name="DejavoWL_settings[continue_url]"
                                           value="<?php echo esc_attr(self::get_setting('continue_url')); ?>"
                                           class="regular-text code"/>
                                    <span class="description"><?php _e('By default, the wishlist popup will only be closed when customers click on the "Continue Shopping" button.', 'dejavo'); ?></span>
                                </td>
                            </tr>
                            <tr class="heading">
                                <th>
                                    <?php _e('Page', 'dejavo'); ?>
                                </th>
                                <td>
                                    <?php _e('Settings for wishlist page.', 'dejavo'); ?>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Wishlist page', 'dejavo'); ?></th>
                                <td>
                                    <?php wp_dropdown_pages([
                                        'selected' => self::get_setting('page_id', ''),
                                        'name' => 'DejavoWL_settings[page_id]',
                                        'show_option_none' => __('Choose a page', 'dejavo'),
                                        'option_none_value' => '',
                                    ]); ?>
                                    <span class="description"><?php printf(__('Add shortcode %s to display the wishlist on a page.', 'dejavo'), '<code>[DejavoWL_list]</code>'); ?></span>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row"><?php _e('Add Wishlist page to My Account', 'dejavo'); ?></th>
                                <td>
                                    <select name="DejavoWL_settings[page_myaccount]">
                                        <option value="yes" <?php selected($page_myaccount, 'yes'); ?>><?php _e('Yes', 'dejavo'); ?></option>
                                        <option value="no" <?php selected($page_myaccount, 'no'); ?>><?php _e('No', 'dejavo'); ?></option>
                                    </select>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row"><?php _e('Action', 'dejavo'); ?></th>
                                <td>
                                    <select name="DejavoWL_settings[menu_action]">
                                        <option value="open_page" <?php selected($menu_action, 'open_page'); ?>><?php _e('Open wishlist page', 'dejavo'); ?></option>
                                        <option value="open_popup" <?php selected($menu_action, 'open_popup'); ?>><?php _e('Open wishlist popup', 'dejavo'); ?></option>
                                    </select>
                                    <span class="description"><?php _e('Action when clicking on the "wishlist menu".', 'dejavo'); ?></span>
                                </td>
                            </tr>

                            <tr class="submit">
                                <th colspan="2">
                                    <?php settings_fields('DejavoWL_settings'); ?><?php submit_button(); ?>
                                </th>
                            </tr>
                        </table>
                    </form>
                </div>
            </div>
            <?php
        }
    }

    function account_items($items)
    {
        if (isset($items['customer-logout'])) {
            $logout = $items['customer-logout'];
            unset($items['customer-logout']);
        } else {
            $logout = '';
        }

        if (!isset($items['wishlist'])) {
            $items['wishlist'] = __('علاقه مندیها', 'dejavo');
        }

        if ($logout) {
            $items['customer-logout'] = $logout;
        }

        return $items;
    }

    function account_endpoint()
    {
        echo do_shortcode('[DejavoWL_list]');
    }

    function enqueue_scripts()
    {

        if (self::get_setting('button_action', 'list') === 'message') {
            wp_enqueue_style('notiny', DEJ_TUI . '/assets/libs/notiny/notiny.css');
            wp_enqueue_script('notiny', DEJ_TUI . '/assets/libs/notiny/notiny.js', ['jquery'], DEJ_VERSION, true);
        }

        // main style
        wp_enqueue_style('DejavoWL-frontend', DEJ_TUI . '/assets/css/wishlist.css', [], DEJ_VERSION);

        // main js
        wp_enqueue_script('DejavoWL-frontend', DEJ_TUI . '/assets/js/wishlist.min.js', [
            'jquery',
            'js-cookie'
        ], DEJ_VERSION, true);

        // localize
        wp_localize_script('DejavoWL-frontend', 'DejavoWL_vars', [
                'ajax_url' => admin_url('admin-ajax.php'),
                'menu_action' => self::get_setting('menu_action', 'open_popup'),
                'wishlist_url' => self::get_url(),
                'button_action' => self::get_setting('button_action', 'list'),
                'message_position' => self::get_setting('message_position', 'right-top'),
                'button_action_added' => self::get_setting('button_action_added', 'popup'),
                'empty_confirm' => __('This action cannot be undone. Are you sure?', 'dejavo'),
                'delete_confirm' => __('This action cannot be undone. Are you sure?', 'dejavo'),
                'copied_text' => __('لینک کپی شد', 'dejavo'),
                'menu_text' => '',
                'button_text' => '<i class="dej-heart"></i>',
                'button_text_added' => '<i class="dej-check"></i>',
            ]
        );
    }

    function get_items($key, $layout = null)
    {
        ob_start();
        // store $global_product
        global $product;
        $global_product = $product;
        $products = self::get_ids($key);
        $link = self::get_setting('link', 'yes');
        $table_tag = $tr_tag = $td_tag = 'div';
        $count = count($products); // count saved products
        $real_count = 0; // count real products
        $real_products = [];

        if ($layout === 'table') {
            $table_tag = 'table';
            $tr_tag = 'tr';
            $td_tag = 'td';
        }


        if (is_array($products) && (count($products) > 0)) {
            echo '<' . $table_tag . ' class="DejavoWL-items">';

            foreach ($products as $product_id => $product_data) {
                global $product;
                $product = wc_get_product($product_id);

                if (!$product || $product->get_status() !== 'publish') {
                    continue;
                }

                if (is_array($product_data) && isset($product_data['time'])) {
                    $product_time = date_i18n(get_option('date_format'), $product_data['time']);
                } else {
                    // for old version
                    $product_time = date_i18n(get_option('date_format'), $product_data);
                }


                echo '<' . $tr_tag . ' class="' . esc_attr('DejavoWL-item DejavoWL-item-' . $product_id) . '" data-id="' . esc_attr($product_id) . '">';

                if ($layout !== 'table') {
                    echo '<div class="DejavoWL-item-inner">';
                }


                if (self::can_edit($key)) {
                    // remove
                    echo '<' . $td_tag . ' class="DejavoWL-item--remove"><span></span></' . $td_tag . '>';
                }

                // image
                echo '<' . $td_tag . ' class="DejavoWL-item--image">';

                if ($link !== 'no') {
                    echo '<a ' . ($link === 'yes_popup' ? 'class="DejavoQV-link" data-id="' . esc_attr($product_id) . '" data-context="DejavoWL"' : '') . ' href="' . esc_url($product->get_permalink()) . '" ' . ($link === 'yes_blank' ? 'target="_blank"' : '') . '>';
                    echo wp_kses_post($product->get_image());
                    echo '</a>';
                } else {
                    echo wp_kses_post($product->get_image());
                }

                echo '</' . $td_tag . '>';

                // info
                echo '<' . $td_tag . ' class="DejavoWL-item--info">';

                if ($link !== 'no') {
                    echo '<div class="DejavoWL-item--name"><a ' . ($link === 'yes_popup' ? 'class="DejavoQV-link" data-id="' . esc_attr($product_id) . '" data-context="DejavoWL"' : '') . ' href="' . esc_url($product->get_permalink()) . '" ' . ($link === 'yes_blank' ? 'target="_blank"' : '') . '>' . wp_kses_post($product->get_name()) . '</a></div>';
                } else {
                    echo '<div class="DejavoWL-item--name">' . wp_kses_post($product->get_name()) . '</div>';
                }

                echo '<div class="DejavoWL-item--price">' . wp_kses_post($product->get_price_html()) . '</div>';

                if (self::get_setting('show_price_change', 'both') !== 'no') {
                    if (isset($product_data['price'])) {
                        $price = $product->get_price();

                        if ($price != $product_data['price']) {
                            // has price change
                            if ($price > $product_data['price']) {
                                // increase
                                $percentage = 100 * ($price - $product_data['price']) / $product_data['price'];
                                $percentage = round($percentage) . '%';
                                $increase = __('Increase {percentage} since added', 'dejavo');
                                $increase_mess = str_replace('{percentage}', $percentage, $increase);

                                if (self::get_setting('show_price_change', 'both') === 'both' || self::get_setting('show_price_change', 'both') === 'increase') {
                                    echo '<div class="DejavoWL-item--price-change DejavoWL-item--price-increase">' . $increase_mess . '</div>';
                                }
                            }

                            if ($price < $product_data['price']) {
                                // decrease
                                $percentage = 100 * ($product_data['price'] - $price) / $product_data['price'];
                                $percentage = round($percentage) . '%';
                                $decrease = esc_html__('Decrease {percentage} since added', 'dejavo');
                                $decrease_mess = str_replace('{percentage}', $percentage, $decrease);

                                if (self::get_setting('show_price_change', 'no') === 'both' || self::get_setting('show_price_change', 'both') === 'decrease') {
                                    echo '<div class="DejavoWL-item--price-change DejavoWL-item--price-decrease">' . $decrease_mess . '</div>';
                                }
                            }
                        }
                    }
                }


                echo '<div class="DejavoWL-item--time">' . esc_html($product_time) . '</div>';
                echo '</' . $td_tag . '>';
                // action
                echo '<' . $td_tag . ' class="DejavoWL-item--actions">';
                echo '<div class="DejavoWL-item--stock">' . wc_get_stock_html($product) . '</div>';
                echo '<div class="DejavoWL-item--add">' . do_shortcode('[add_to_cart style="" show_price="false" id="' . esc_attr($product_id) . '"]') . '</div>';
                echo '</' . $td_tag . '>';


                if ($layout !== 'table') {
                    echo '</div><!-- /DejavoWL-item-inner -->';
                }

                echo '</' . $tr_tag . '>';

                $real_products[$product_id] = $product_data;
                $real_count++;

            }

            echo '</' . $table_tag . '>';
        } else {
            echo '<div class="DejavoWL-popup-content-mid-message">' . __('There are no products on the Wishlist!', 'dejavo') . '</div>';
        }

        // restore $global_product
        $product = $global_product;

        // update products
        if ($real_count < $count) {
            update_option('DejavoWL_list_' . $key, $real_products);
        }

        return ob_get_clean();
    }

    function wp_footer()
    {
        if (is_admin()) {
            return;
        }

        echo '<div id="DejavoWL_wishlist" class="DejavoWL-popup ' . esc_attr('DejavoWL-popup-' . self::get_setting('popup_position', 'center')) . '"></div>';
    }

    function wishlist_content($key = false, $message = '')
    {
        if (empty($key)) {
            $key = self::get_key();
        }

        $products = self::get_ids($key);
        $count = count($products);
        $name = __('Wishlist', 'dejavo');

        ob_start();
        ?>
        <div class="DejavoWL-popup-inner" data-key="<?php echo esc_attr($key); ?>">
            <div class="DejavoWL-popup-content">
                <div class="DejavoWL-popup-content-top">
                    <span class="DejavoWL-name"><?php echo esc_html($name); ?></span>
                    <?php
                    echo '<span class="DejavoWL-count-wrapper">';
                    echo '<span class="DejavoWL-count">' . esc_html($count) . '</span>';

                    if (self::get_setting('empty_button', 'yes') === 'yes') {
                        echo '<span class="DejavoWL-empty"' . ($count ? '' : ' style="display:none"') . '>' . __('remove all', 'dejavo') . '</span>';
                    }

                    echo '</span>';
                    ?>
                    <span class="DejavoWL-popup-close"></span>
                </div>
                <div class="DejavoWL-popup-content-mid">
                    <?php if (!empty($message)) {
                        echo '<div class="DejavoWL-popup-content-mid-message">' . esc_html($message) . '</div>';
                    } else {
                        echo self::get_items($key);
                    } ?>
                </div>
                <div class="DejavoWL-popup-content-bot">
                    <div class="DejavoWL-popup-content-bot-inner">
                        <a class="DejavoWL-page" href="<?php echo esc_url(self::get_url($key, true)); ?>">
                            <?php echo __('Open wishlist page', 'dejavo'); ?>
                        </a>
                        <a class="DejavoWL-continue"
                           href="<?php echo esc_url(self::get_setting('continue_url')); ?>"
                           data-url="<?php echo esc_url(self::get_setting('continue_url')); ?>">
                            <?php echo __('Continue shopping', 'dejavo'); ?>
                        </a>
                    </div>
                    <div class="DejavoWL-notice"></div>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }


    function update_product_count($product_id, $action = 'add')
    {
        $meta_count = 'DejavoWL_count';
        $meta_time = ($action === 'add' ? 'DejavoWL_add' : 'DejavoWL_remove');
        $count = get_post_meta($product_id, $meta_count, true);
        $new_count = 0;

        if ($action === 'add') {
            if ($count) {
                $new_count = absint($count) + 1;
            } else {
                $new_count = 1;
            }
        } elseif ($action === 'remove') {
            if ($count && (absint($count) > 1)) {
                $new_count = absint($count) - 1;
            } else {
                $new_count = 0;
            }
        }

        update_post_meta($product_id, $meta_count, $new_count);
        update_post_meta($product_id, $meta_time, time());
    }

    public static function generate_key()
    {
        $key = '';
        $key_str = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        $key_str_len = strlen($key_str);

        for ($i = 0; $i < 6; $i++) {
            $key .= $key_str[random_int(0, $key_str_len - 1)];
        }

        return $key;
    }

    public static function can_edit($key)
    {
        if (is_user_logged_in()) {
            if (get_user_meta(get_current_user_id(), 'DejavoWL_key', true) === $key) {
                return true;
            }

            if (($keys = get_user_meta(get_current_user_id(), 'DejavoWL_keys', true)) && isset($keys[$key])) {
                return true;
            }
        } else {
            if (isset($_COOKIE['DejavoWL_key']) && (sanitize_text_field($_COOKIE['DejavoWL_key']) === $key)) {
                return true;
            }
        }

        return false;
    }

    public static function get_page_id()
    {
        if (self::get_setting('page_id')) {
            return absint(self::get_setting('page_id'));
        }

        return false;
    }

    public static function get_key($new = false)
    {
        if ($new) {
            // get a new key for multiple wishlist
            $key = self::generate_key();

            while (self::exists_key($key)) {
                $key = self::generate_key();
            }

            return $key;
        } else {
            if (!is_user_logged_in() && (self::get_setting('disable_unauthenticated', 'no') === 'yes')) {
                return '#';
            }

            if (is_user_logged_in() && (($user_id = get_current_user_id()) > 0)) {
                $key = get_user_meta($user_id, 'DejavoWL_key', true);

                if (empty($key)) {
                    $key = self::generate_key();

                    while (self::exists_key($key)) {
                        $key = self::generate_key();
                    }

                    // set a new key
                    update_user_meta($user_id, 'DejavoWL_key', $key);

                    // multiple wishlist
                    update_user_meta($user_id, 'DejavoWL_keys', [
                        $key => [
                            'type' => 'primary',
                            'name' => '',
                            'time' => ''
                        ]
                    ]);
                }

                return $key;
            }

            if (isset($_COOKIE['DejavoWL_key'])) {
                return trim(sanitize_text_field($_COOKIE['DejavoWL_key']));
            }

            return 'DejavoWL';
        }
    }

    public static function exists_key($key)
    {
        if (get_option('DejavoWL_list_' . $key)) {
            return true;
        }

        return false;
    }

    public static function get_ids($key = null)
    {
        if (!$key) {
            $key = self::get_key();
        }

        return (array)get_option('DejavoWL_list_' . $key, []);
    }

    public static function get_products()
    {
        return self::$products;
    }

    public static function get_url($key = null, $full = false)
    {
        $url = home_url('/');

        if ($page_id = self::get_page_id()) {
            if ($full) {
                if (!$key) {
                    $key = self::get_key();
                }

                if (get_option('permalink_structure') !== '') {
                    $url = trailingslashit(get_permalink($page_id)) . $key;
                } else {
                    $url = get_permalink($page_id) . '&DejavoWL_id=' . $key;
                }
            } else {
                $url = get_permalink($page_id);
            }
        }

        return esc_url($url);
    }

    public static function get_count($key = null)
    {
        if (!$key) {
            $key = self::get_key();
        }

        $products = self::get_ids($key);
        $count = count($products);

        return esc_html($count);
    }

    function wp_login($user_login, $user)
    {
        if (isset($user->data->ID)) {
            $key = get_user_meta($user->data->ID, 'DejavoWL_key', true);

            if (empty($key)) {
                $key = self::generate_key();

                while (self::exists_key($key)) {
                    $key = self::generate_key();
                }

                // set a new key
                update_user_meta($user->data->ID, 'DejavoWL_key', $key);
            }

            // multiple wishlist
            if (!get_user_meta($user->data->ID, 'DejavoWL_keys', true)) {
                update_user_meta($user->data->ID, 'DejavoWL_keys', [
                    $key => [
                        'type' => 'primary',
                        'name' => '',
                        'time' => ''
                    ]
                ]);
            }

            $secure = wc_site_is_https() && is_ssl();
            $httponly = false;

            if (isset($_COOKIE['DejavoWL_key']) && !empty($_COOKIE['DejavoWL_key'])) {
                wc_setcookie('DejavoWL_key_ori', trim(sanitize_text_field($_COOKIE['DejavoWL_key'])), time() + 604800, $secure, $httponly);
            }

            wc_setcookie('DejavoWL_key', $key, time() + 604800, $secure, $httponly);
        }
    }

    function wp_logout($user_id)
    {
        if (isset($_COOKIE['DejavoWL_key_ori']) && !empty($_COOKIE['DejavoWL_key_ori'])) {
            $secure = wc_site_is_https() && is_ssl();
            $httponly = false;

            wc_setcookie('DejavoWL_key', trim(sanitize_text_field($_COOKIE['DejavoWL_key_ori'])), time() + 604800, $secure, $httponly);
        } else {
            unset($_COOKIE['DejavoWL_key_ori']);
            unset($_COOKIE['DejavoWL_key']);
        }
    }


    function wishlist_quickview()
    {
        if (!check_ajax_referer('DejavoWL-nonce', 'nonce', false)) {
            die(__('Permissions check failed!', 'dejavo'));
        }

        global $wpdb;
        ob_start();
        echo '<div class="DejavoWL-quickview-items">';

        if (isset($_POST['key']) && $_POST['key'] != '') {
            $key = sanitize_text_field($_POST['key']);
            $products = self::get_ids($key);
            $count = count($products);

            if (count($products) > 0) {
                $user = $wpdb->get_results($wpdb->prepare('SELECT user_id FROM `' . $wpdb->prefix . 'usermeta` WHERE `meta_key` = "DejavoWL_keys" AND `meta_value` LIKE "%s" LIMIT 1', '%"' . $key . '"%'));

                echo '<div class="DejavoWL-quickview-item">';
                echo '<div class="DejavoWL-quickview-item-image"><a href="' . esc_url(self::get_url($key, true)) . '" target="_blank">' . $key . '</a></div>';
                echo '<div class="DejavoWL-quickview-item-info">';

                if (!empty($user)) {
                    $user_id = $user[0]->user_id;
                    $user_data = get_userdata($user_id);

                    echo '<div class="DejavoWL-quickview-item-title"><a href="#" class="DejavoWL_action" data-uid="' . esc_attr($user_id) . '">' . $user_data->user_login . '</a></div>';
                    echo '<div class="DejavoWL-quickview-item-data">' . $user_data->user_email . ' | ' . sprintf(_n('%s product', '%s products', $count, 'dejavo'), number_format_i18n($count)) . '</div>';
                } else {
                    echo '<div class="DejavoWL-quickview-item-title">' . __('Guest', 'dejavo') . '</div>';
                    echo '<div class="DejavoWL-quickview-item-data">' . sprintf(_n('%s product', '%s products', $count, 'dejavo'), number_format_i18n($count)) . '</div>';
                }

                echo '</div><!-- /DejavoWL-quickview-item-info -->';
                echo '</div><!-- /DejavoWL-quickview-item -->';

                foreach ($products as $pid => $data) {
                    if ($_product = wc_get_product($pid)) {
                        echo '<div class="DejavoWL-quickview-item">';
                        echo '<div class="DejavoWL-quickview-item-image">' . $_product->get_image() . '</div>';
                        echo '<div class="DejavoWL-quickview-item-info">';
                        echo '<div class="DejavoWL-quickview-item-title"><a href="' . get_edit_post_link($pid) . '" target="_blank">' . $_product->get_name() . '</a></div>';
                        echo '<div class="DejavoWL-quickview-item-data">' . date_i18n(get_option('date_format'), $data['time']) . ' <span class="DejavoWL-quickview-item-links">| ' . sprintf(__('Product ID: %s', 'dejavo'), $pid) . ' | <a href="#" class="DejavoWL_action" data-pid="' . esc_attr($pid) . '">' . __('See in wishlist', 'dejavo') . '</a></span></div>';
                        echo '</div><!-- /DejavoWL-quickview-item-info -->';
                        echo '</div><!-- /DejavoWL-quickview-item -->';
                    } else {
                        echo '<div class="DejavoWL-quickview-item">';
                        echo '<div class="DejavoWL-quickview-item-image">' . wc_placeholder_img() . '</div>';
                        echo '<div class="DejavoWL-quickview-item-info">';
                        echo '<div class="DejavoWL-quickview-item-title">' . sprintf(esc_html__('Product ID: %s', 'dejavo'), $pid) . '</div>';
                        echo '<div class="DejavoWL-quickview-item-data">' . __('This product is not available!', 'dejavo') . '</div>';
                        echo '</div><!-- /DejavoWL-quickview-item-info -->';
                        echo '</div><!-- /DejavoWL-quickview-item -->';
                    }
                }
            } else {
                echo '<div class="DejavoWL-quickview-item">';
                echo '<div class="DejavoWL-quickview-item-image">' . wc_placeholder_img() . '</div>';
                echo '<div class="DejavoWL-quickview-item-info">';
                echo '<div class="DejavoWL-quickview-item-title">' . sprintf(esc_html__('Wishlist #%s', 'dejavo'), $key) . '</div>';
                echo '<div class="DejavoWL-quickview-item-data">' . __('This wishlist have no product!', 'dejavo') . '</div>';
                echo '</div><!-- /DejavoWL-quickview-item-info -->';
                echo '</div><!-- /DejavoWL-quickview-item -->';
            }
        } elseif (isset($_POST['pid'])) {
            $pid = (int)sanitize_text_field($_POST['pid']);
            $per_page = 10;
            $page = isset($_POST['page']) ? abs((int)$_POST['page']) : 1;
            $offset = ($page - 1) * $per_page;
            $total = $wpdb->get_var($wpdb->prepare('SELECT COUNT(*) FROM `' . $wpdb->prefix . 'options` WHERE `option_name` LIKE "%DejavoWL_list_%" AND `option_value` LIKE "%s"', '%i:' . $pid . ';%'));
            $keys = $wpdb->get_results($wpdb->prepare('SELECT option_name FROM `' . $wpdb->prefix . 'options` WHERE `option_name` LIKE "%DejavoWL_list_%" AND `option_value` LIKE "%s" limit ' . $per_page . ' offset ' . $offset, '%i:' . $pid . ';%'));

            if ($total && is_countable($keys) && count($keys)) {
                echo '<div class="DejavoWL-quickview-item">';

                if ($_product = wc_get_product($pid)) {
                    echo '<div class="DejavoWL-quickview-item-image">' . $_product->get_image() . '</div>';
                    echo '<div class="DejavoWL-quickview-item-info">';
                    echo '<div class="DejavoWL-quickview-item-title"><a href="' . get_edit_post_link($pid) . '" target="_blank">' . $_product->get_name() . '</a></div>';
                    echo '<div class="DejavoWL-quickview-item-data">' . sprintf(esc_html__('Product ID: %s', 'dejavo'), $pid) . ' | ' . sprintf(_n('%s wishlist', '%s wishlists', $total, 'DejavoWL'), number_format_i18n($total)) . '</div>';
                } else {
                    echo '<div class="DejavoWL-quickview-item-image">' . wc_placeholder_img() . '</div>';
                    echo '<div class="DejavoWL-quickview-item-info">';
                    echo '<div class="DejavoWL-quickview-item-title">' . sprintf(esc_html__('Product ID: %s', 'dejavo'), $pid) . '</div>';
                    echo '<div class="DejavoWL-quickview-item-data">' . __('This product is not available!', 'dejavo') . '</div>';
                }

                // paging
                if ($total > $per_page) {
                    $pages = ceil($total / $per_page);
                    echo '<div class="DejavoWL-quickview-item-paging">Page ';

                    echo '<select class="DejavoWL_paging" data-pid="' . $pid . '">';

                    for ($i = 1; $i <= $pages; $i++) {
                        echo '<option value="' . $i . '" ' . selected($page, $i, false) . '>' . $i . '</option>';
                    }

                    echo '</select> / ' . $pages;

                    echo '</div><!-- /DejavoWL-quickview-item-paging -->';
                }

                echo '</div><!-- /DejavoWL-quickview-item-info -->';
                echo '</div><!-- /DejavoWL-quickview-item -->';

                foreach ($keys as $item) {
                    $products = get_option($item->option_name);
                    $products_count = count($products);
                    $key = str_replace('DejavoWL_list_', '', $item->option_name);
                    $user = $wpdb->get_results($wpdb->prepare('SELECT user_id FROM `' . $wpdb->prefix . 'usermeta` WHERE `meta_key` = "DejavoWL_keys" AND `meta_value` LIKE "%s" LIMIT 1', '%"' . $key . '"%'));

                    echo '<div class="DejavoWL-quickview-item">';
                    echo '<div class="DejavoWL-quickview-item-image"><a href="' . esc_url(self::get_url($key, true)) . '" target="_blank">' . esc_html($key) . '</a></div>';
                    echo '<div class="DejavoWL-quickview-item-info">';

                    if (!empty($user)) {
                        $user_id = $user[0]->user_id;
                        $user_data = get_userdata($user_id);

                        echo '<div class="DejavoWL-quickview-item-title"><a href="#" class="DejavoWL_action" data-uid="' . esc_attr($user_id) . '">' . $user_data->user_login . '</a></div>';
                        echo '<div class="DejavoWL-quickview-item-data">' . $user_data->user_email . '  | <a href="#" class="DejavoWL_action DejavoWL_action_' . $products_count . '" data-key="' . esc_attr($key) . '">' . sprintf(_n('%s product', '%s products', $products_count, 'dejavo'), number_format_i18n($products_count)) . '</a></div>';
                    } else {
                        echo '<div class="DejavoWL-quickview-item-title">' . __('Guest', 'dejavo') . '</div>';
                        echo '<div class="DejavoWL-quickview-item-data"><a href="#" class="DejavoWL_action" data-key="' . esc_attr($key) . '">' . sprintf(_n('%s product', '%s products', $products_count, 'dejavo'), number_format_i18n($products_count)) . '</a></div>';
                    }

                    echo '</div><!-- /DejavoWL-quickview-item-info -->';
                    echo '</div><!-- /DejavoWL-quickview-item -->';
                }
            }
        } elseif (isset($_POST['uid'])) {
            $user_id = (int)sanitize_text_field($_POST['uid']);
            $keys = get_user_meta($user_id, 'DejavoWL_keys', true) ?: [];

            if ($user = get_user_by('id', $user_id)) {
                echo '<div class="DejavoWL-quickview-item">';
                echo '<div class="DejavoWL-quickview-item-image"><img src="' . esc_url(get_avatar_url($user_id)) . '" /></div>';
                echo '<div class="DejavoWL-quickview-item-info">';
                echo '<div class="DejavoWL-quickview-item-title"><a href="' . get_edit_user_link($user_id) . '" target="_blank">' . $user->user_login . '</a></div>';
                echo '<div class="DejavoWL-quickview-item-data">' . $user->user_email . '</div>';
                echo '</div><!-- /DejavoWL-quickview-item-info -->';
                echo '</div><!-- /DejavoWL-quickview-item -->';
            }

            if (is_array($keys) && count($keys)) {
                foreach ($keys as $key => $data) {
                    $products = self::get_ids($key);
                    $products_count = count($products);

                    echo '<div class="DejavoWL-quickview-item">';
                    echo '<div class="DejavoWL-quickview-item-image"><a href="' . esc_url(self::get_url($key, true)) . '" target="_blank">' . $key . '</a></div>';
                    echo '<div class="DejavoWL-quickview-item-info">';
                    echo '<div class="DejavoWL-quickview-item-title">' . (!empty($data['name']) ? $data['name'] : 'Primary') . '</div>';
                    echo '<div class="DejavoWL-quickview-item-data"><a href="#" class="DejavoWL_action DejavoWL_action_' . $products_count . '" data-key="' . esc_attr($key) . '">' . sprintf(_n('%s product', '%s products', $products_count, 'dejavo'), number_format_i18n($products_count)) . '</a></div>';
                    echo '</div><!-- /DejavoWL-quickview-item-info -->';
                    echo '</div><!-- /DejavoWL-quickview-item -->';
                }
            }
        }

        echo '</div><!-- /DejavoWL-quickview-items -->';
        echo ob_get_clean();
        die();
    }


    function get_data()
    {
        $data = [
            'key' => self::get_key(),
            'ids' => self::get_ids(),
        ];

        wp_send_json($data);
    }
}
return DejavoWL::instance();
}




